<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\WithdrawalCategory;
use Illuminate\Support\Facades\Storage;
use App\Models\WithdrawalSubCategory;
use App\Models\WithdrawalHistory;
use App\Lib\FileManagement;


class GatewaysController extends Controller
{
    private $fileManager;

    public function __construct(FileManagement $fileManager)
    {
        $this->fileManager = $fileManager;
    }

    public function index()
    {
        $withdrawalsCategories = WithdrawalCategory::all();
        return view('admin.gateways', ['withdrawalsCategories' => $withdrawalsCategories]);
    }


    public function store(Request $request)
    {

        $validatedData = $this->validateStoreRequest($request);
        if ($this->categoryExists($validatedData['name'])) {
            return redirect()->back()->with('error', 'Withdrawal category already exists.');
        }

        $coverPath = $this->storeCover($request);
        $rewardImage = $this->storeRewardImage($request);

        $category = WithdrawalCategory::create([
            'name' => $validatedData['name'],
            'withdrawal_type' => $validatedData['withdrawal_type'],
            'cover' => $coverPath,
            'reward_img' => $rewardImage,
            'bg_color' => $validatedData['bg_color'],
        ]);


        $this->createSubCategories($category->id);

        return redirect()->back()->with('success', 'Withdrawal category added successfully.');
    }

    public function update(Request $request, $id)
    {
        $validatedData = $this->validateUpdateRequest($request);

        $category = WithdrawalCategory::findOrFail($id);

        if ($this->categoryNameChanged($category, $validatedData['name'])) {
            if ($this->categoryExists($validatedData['name'])) {
                return redirect()->back()->with('error', 'Withdrawal category name already exists.');
            }
            $category->name = $validatedData['name'];
        }

        if ($request->hasFile('cover')) {
            $this->updateCover($category, $request);
        }

        if ($request->hasFile('reward_img')) {
            $this->updateRewardImage($category, $request);
        }


        $category->bg_color = $validatedData['bg_color'] ?? $category->bg_color;
        $category->save();

        return redirect()->back()->with('success', 'Withdrawal category updated successfully.');
    }

    public function destroy($id)
    {
        $category = WithdrawalCategory::findOrFail($id);

        $this->deleteCover($category);
        $this->deleteRewardImage($category);

        $category->delete();

        return redirect()->back()->with('success', 'Withdrawal category deleted successfully.');
    }

    // Private Methods

    private function validateStoreRequest(Request $request)
    {
        return $request->validate([
            'name' => 'required|string|max:255',
            'withdrawal_type' => 'required|in:cash,gift_card,skins',
            'cover' => 'required|image|mimes:jpeg,png,jpg,svg,webp|max:1024',
            'reward_img' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:1024',
            'bg_color' => 'nullable|string|max:255',
        ]);
    }

    private function validateUpdateRequest(Request $request)
    {
        return $request->validate([
            'name' => 'required|string|max:255',
            'withdrawal_type' => 'required|in:cash,gift_card,skins',
            'cover' => 'nullable|image|mimes:jpeg,png,jpg,svg,webp|max:1024',
            'reward_img' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:1024',
            'bg_color' => 'nullable|string|max:255',
        ]);
    }

    private function categoryExists($name)
    {
        return WithdrawalCategory::where('name', $name)->exists();
    }

    private function storeCover(Request $request)
    {
        return $request->hasFile('cover')
            ? $this->fileManager->uploadImage($request->file('cover'), 'gateways')
            : null; // Return null if no file was uploaded
    }


    private function updateCover($category, Request $request)
    {
        if ($category->cover) {
            $this->fileManager->delete($category->cover);
        }

        $category->cover = $this->fileManager->uploadImage($request->file('cover'), 'gateways');
    }



    private function deleteCover($category)
    {
        if ($category->cover) {
            $this->fileManager->delete($category->cover);
        }
    }


    private function storeRewardImage(Request $request)
    {
        return $request->hasFile('reward_img')
            ? $this->fileManager->uploadImage($request->file('reward_img'), 'reward')
            : null;
    }


    private function updateRewardImage($category, Request $request)
    {
        if ($category->reward_img) {
            $this->fileManager->delete($category->reward_img);
        }

        $category->reward_img = $this->fileManager->uploadImage($request->file('reward_img'), 'reward');
    }

    private function deleteRewardImage($category)
    {
        if ($category->reward_img) {
            $this->fileManager->delete($category->reward_img);
        }
    }

    private function categoryNameChanged($category, $name)
    {
        return $category->name !== $name;
    }

    private function createSubCategories($categoryId)
    {
        $amounts = [5, 10, 15, 20, 50, 100];

        foreach ($amounts as $amount) {
            WithdrawalSubCategory::create([
                'amount' => $amount,
                'withdrawal_categories_id' => $categoryId,
            ]);
        }
    }


    //items Controllers
    public function addItems($categoryId)
    {
        $withdrawalsCategory = WithdrawalCategory::findOrFail($categoryId);

        $withdrawalsSubCategories = WithdrawalSubCategory::where('withdrawal_categories_id', $categoryId)
            ->get();

        return view('admin.gateways-items', [
            'withdrawalsCategory' => $withdrawalsCategory,
            'withdrawalsSubCategories' => $withdrawalsSubCategories,
        ]);
    }



    public function storeItems($categoryId, Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|between:0.01,999999.99',
        ]);

        $existingItem = WithdrawalSubCategory::where('withdrawal_categories_id', $categoryId)
            ->where('amount', $request->amount)
            ->exists();

        if ($existingItem) {
            return redirect()->back()->with('error', 'This amount already exists for this category.');
        }

        WithdrawalSubCategory::create([
            'amount' => $request->amount,
            'withdrawal_categories_id' => $categoryId,
        ]);

        return redirect()->back();
    }

    public function deleteItem($id)
    {
        $item = WithdrawalSubCategory::findOrFail($id);
        $item->delete();

        return redirect()->back();
    }

    public function withdraw(Request $request)
    {
        $search = str_replace(' ', '', $request->input('search') ?? '');
        $perPage = $request->input('perPage', 15);

        $withdrawalsQuery = WithdrawalHistory::with('category', 'user')
            ->where('status', 'pending');

        if (!empty($search)) {
            $withdrawalsQuery->where(function ($query) use ($search) {
                $query->where('redeem_wallet', $search)
                    ->orWhereHas('user', function ($query) use ($search) {
                        $query->where('uid', $search);
                    });
            });

            $userPagination = $withdrawalsQuery->count();
            $withdrawals = $withdrawalsQuery->paginate($userPagination);
        } else {
            $withdrawals = $withdrawalsQuery->paginate($perPage);
            $userPagination = $perPage;
        }

        $totals = WithdrawalHistory::selectRaw('COUNT(*) as totalPendingWithdrawals, SUM(amount) as totalUSDAmount')
            ->where('status', 'pending')
            ->when(!empty($search), function ($query) use ($search) {
                $query->where(function ($subQuery) use ($search) {
                    $subQuery->where('redeem_wallet', $search)
                        ->orWhereHas('user', function ($subQuery) use ($search) {
                            $subQuery->where('uid', $search);
                        });
                });
            })
            ->first();

        $userTotal = !empty($search)
            ? WithdrawalHistory::where('redeem_wallet', $search)
                ->where('status', 'pending')
                ->sum('amount')
            : 0;

        $totalPendingWithdrawals = $totals->totalPendingWithdrawals ?? 0;
        $totalUSDAmount = $totals->totalUSDAmount ?? 0;

        return view('admin.withdraw', compact('withdrawals', 'totalPendingWithdrawals', 'totalUSDAmount', 'userTotal'));
    }
    public function markCompleted(Request $request)
    {
        if ($request->has('selectedWithdrawals')) {
            $selectedWithdrawals = $request->input('selectedWithdrawals');
            WithdrawalHistory::whereIn('id', $selectedWithdrawals)
                ->update(['status' => 'completed']);

            return redirect()->back()->with('success', 'Withdrawals marked as completed successfully!');
        }

        return redirect()->back()->with('error', 'Please select at least one withdrawal to mark as completed.');
    }

    public function CancelWithdraw(Request $request)
    {
        $withdrawalId = $request->input('withdraw-id');
        $withdrawal = WithdrawalHistory::findOrFail($withdrawalId);
        $user = $withdrawal->user;

        if ($request->has('confirmCancel')) {
            $user->balance += $withdrawal->amount;
            $user->save();

            $withdrawal->status = 'refunded';
        } else {
            $withdrawal->status = 'rejected';
        }

        $withdrawal->save();

        return redirect()->back()->with('success', 'Withdrawal status updated successfully.');
    }




}
