<?php

namespace App\Http\Controllers\Admin\Networks;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Cache;
use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Models\Network;
use App\Lib\FileManagement;


class OffersController extends Controller
{
    private $fileManager;

    public function __construct(FileManagement $fileManager)
    {
        $this->fileManager = $fileManager;
    }

    public function index()
    {
        $networks = Network::all();

        $networks->transform(function ($network) {
            $postbackUrl = $this->generatePostbackUrl($network);

            $network->postbackUrl = $postbackUrl;

            return $network;
        });

        return view('admin.networks.index', compact('networks'));
    }

    private function generatePostbackUrl($network)
    {
        $postbackUrl = route('postback.callback', [
            'param_secret' => $network->param_secret,
            'network_slug' => $network->network_slug,
        ]);

        $params = [];

        if ($network->param_url_visibility == 1) {
            $params = [
                'payout' => $network->param_payout,
                'uid' => $network->param_user_id,
                'of_id' => $network->param_offer_id,
                'of_name' => $network->param_offer_name,
                'tx_id' => $network->param_tx_id,
                'reward' => $network->param_amount,
                'ip' => $network->param_ip,
                'country' => $network->param_country,
                'status' => $network->param_status,
            ];

            $params = array_filter($params, function ($value) {
                return !is_null($value);
            });

            // Convert to query string format
            $params = array_map(function ($key, $value) {
                return $key . '=' . $value;
            }, array_keys($params), $params);
        }


        if (!empty($params)) {
            $queryString = implode('&', $params);
            $postbackUrl .= '?' . $queryString;
        }

        return $postbackUrl;
    }

    public function ApiOffers()
    {
        $settings = [
            'ogads_api_key',
            'ogads_rate',
            'ogads_status_enabled',
            'adgate_api_key',
            'adgate_wall',
            'adgate_rate',
            'adgate_affiliate_id',
            'adgate_status_enabled',
            'notik_app_id',
            'notik_pub_id',
            'notik_api_key',
            'notik_api_key_secret',
            'notik_rate',
            'notik_status_enabled',
        ];

        $settingsData = Setting::whereIn('name', $settings)->pluck('value', 'name');

        return view('admin.networks.api-offers', compact('settingsData'));
    }

    public function StoreApi(Request $request)
    {
        $validatedData = $request->validate([
            'ogads_api_key' => 'nullable|string|max:1024',
            'ogads_rate' => 'nullable|numeric',
            'ogads_status_enabled' => 'required|in:0,1',
            'adgate_api_key' => 'nullable|string|max:1024',
            'adgate_wall' => 'nullable|string|max:1024',
            'adgate_affiliate_id' => 'nullable|string|max:1024',
            'adgate_rate' => 'nullable|numeric',
            'adgate_status_enabled' => 'required|in:0,1',
            'notik_app_id' => 'required|string|max:50',
            'notik_pub_id' => 'required|string|max:191',
            'notik_api_key' => 'required|string|max:255',
            'notik_api_key_secret' => 'required|string|max:255',
            'notik_rate' => 'nullable|numeric',
            'notik_status_enabled' => 'required|in:0,1',

        ]);


        $settings = [
            'ogads_api_key' => $validatedData['ogads_api_key'],
            'ogads_rate' => $validatedData['ogads_rate'],
            'ogads_status_enabled' => $validatedData['ogads_status_enabled'],
            'adgate_api_key' => $validatedData['adgate_api_key'],
            'adgate_wall' => $validatedData['adgate_wall'],
            'adgate_affiliate_id' => $validatedData['adgate_affiliate_id'],
            'adgate_rate' => $validatedData['adgate_rate'],
            'adgate_status_enabled' => $validatedData['adgate_status_enabled'],
            'notik_app_id' => $validatedData['notik_app_id'],
            'notik_pub_id' => $validatedData['notik_pub_id'],
            'notik_api_key' => $validatedData['notik_api_key'],
            'notik_api_key_secret' => $validatedData['notik_api_key_secret'],
            'notik_rate' => $validatedData['notik_rate'],
            'notik_status_enabled' => $validatedData['notik_status_enabled'],
        ];


        foreach ($settings as $name => $value) {
            Setting::updateOrCreate(
                ['name' => $name],
                ['value' => $value]
            );
        }
        Cache::forget('apiSettings');


        return redirect()->back()->with('success', 'Settings saved successfully.');
    }


    public function newNetwork()
    {
        return view('admin.networks.new-network');
    }

    public function storeNetwork(Request $request)
    {
        // Normalize network name to lowercase and validate the request
        $request->merge(['network_name' => strtolower($request->input('network_name'))]);
        $validatedData = $request->validate($this->networkValidationRules());

        // Check if the network name already exists
        if (Network::where('network_name', $validatedData['network_name'])->exists()) {
            return redirect()->back()->withInput()->withErrors(['network_name' => 'The network name already exists.']);
        }

        try {
            // Prepare and create a new network record
            $validatedData = $this->prepareNetworkData($request, $validatedData);
            Network::create($validatedData);
            return redirect()->route('admin.networks.index')->with('success', 'Network added successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Failed to add network. Please try again.');
        }
    }

    public function editNetwork($id)
    {
        $networkInfo = Network::find($id);

        if (!$networkInfo) {
            return redirect()->route('admin.networks.index')->with('error', 'Network not found.');
        }

        return view('admin.networks.edit-network', compact('networkInfo'));
    }

    public function storeEditNetwork(Request $request, $id)
    {
        $request->validate($this->networkValidationRules());

        try {
            $network = Network::findOrFail($id);
            $network->fill($request->except('network_image'));

            if ($request->filled('network_name')) {
                $network->network_slug = $this->generateSlug($request->input('network_name'));
            }

            if ($request->hasFile('network_image')) {
                if ($network instanceof Network) {
                    $this->updateNetworkImage($network, $request->file('network_image'));
                }
            }

            $network->save();
            return redirect()->route('admin.networks.index')->with('success', 'Network updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to update network. Please try again.');
        }
    }

    public function deleteNetwork($id)
    {
        try {
            $network = Network::findOrFail($id);

            if ($network->network_image) {
                $this->fileManager->delete($network->network_image);
            }

            $network->delete();
            return redirect()->route('admin.networks.index')->with('success', 'Network deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to delete network. Please try again.');
        }
    }


    private function networkValidationRules(): array
    {
        return [
            'network_name' => 'required|string|max:50',
            'offerwall_url' => 'required|string',
            'network_description' => 'required|string|max:50',
            'network_rating' => 'required|integer|min:1|max:5',
            'network_image' => 'nullable|image|max:2048',
            'network_status' => 'required|in:1,0',
            'network_type' => 'required|in:offer,survey',
            'postback_method' => 'required|in:GET,POST',
            'param_url_visibility' => 'required|in:1,0',
            'param_custom_rate' => 'nullable|numeric|required_without:param_amount',
            'param_amount' => 'nullable|string|max:40|required_without:param_custom_rate',
            'param_payout' => 'nullable|string|max:40',
            'param_user_id' => 'required|string|max:40',
            'param_offer_id' => 'nullable|string|max:40',
            'param_offer_name' => 'nullable|string|max:40',
            'param_tx_id' => 'nullable|string|max:40',
            'param_ip' => 'nullable|string|max:40',
            'param_country' => 'nullable|string',
            'param_status' => 'nullable|string',
        ];
    }

    private function prepareNetworkData(Request $request, array $validatedData): array
    {
        $validatedData['network_slug'] = $this->generateSlug($validatedData['network_name']);
        $validatedData['param_secret'] = Str::random(10);

        if ($request->hasFile('network_image')) {
            $validatedData['network_image'] = $this->fileManager->uploadImage(
                $request->file('network_image'),
                'networks'
            );
        }

        return $validatedData;
    }

    private function generateSlug(string $networkName): string
    {
        return strtolower(str_replace(' ', '-', $networkName));
    }

    private function updateNetworkImage(Network $network, $imageFile): void
    {
        if ($network->network_image) {
            $this->fileManager->delete($network->network_image);
        }

        if ($newImagePath = $this->fileManager->uploadImage($imageFile, 'networks')) {
            $network->network_image = $newImagePath;
        }
    }






}
