<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;
use App\Models\Setting;
use App\Lib\FileManagement;

class SettingController extends Controller
{
    private $fraudSettings = [
        'email_verification_enabled',
        'registration_enabled',
        'vpn_ban_enabled',
        'country_ban_enabled',
        'one_device_registration_only',
        'vpn_detection_enabled',
        'temporary_email_ban_enabled',
        'captcha_enabled'
    ];

    private $generalSettings = [
        'site_name',
        'site_currency_symbol',
        'active_template',
        'referral_percentage',
        'site_logo',
        'site_favicon',
        'seo_description',
        'seo_keywords',
        'google_analytics_key',
        'proxycheck_api_key',
    ];

    private $fileManager;

    public function __construct(FileManagement $fileManager)
    {
        $this->fileManager = $fileManager;
    }

    public function frauds()
    {
        $settingsFraud = Setting::whereIn('name', $this->fraudSettings)->pluck('value', 'name');

        return view('admin.frauds', compact('settingsFraud'));
    }

    public function updateFraud(Request $request)
    {
        $this->validateFraud($request);

        $settings = $this->getSettingsFromRequest($request, $this->fraudSettings)
            ->mapWithKeys(fn($value, $key) => [$key => (int) $value]);

        $settings->each(fn($value, $name) => Setting::updateOrCreate(['name' => $name], ['value' => $value]));

        return redirect()->back()->with('success', 'Fraud settings updated successfully.');
    }

    public function settings()
    {
        $templates = get_available_templates();
        $systemSettings = Setting::whereIn('name', $this->generalSettings)->pluck('value', 'name');

        return view('admin.settings', compact('systemSettings', 'templates'));
    }


    public function updateSettings(Request $request)
    {
        $this->validateSettings($request);

        $data = $this->getSettingsFromRequest($request, $this->generalSettings)->toArray();

        if ($request->hasFile('site_logo')) {
            $this->deleteOldFile('site_logo', 'assets/images/logo');

            $data['site_logo'] = $this->fileManager->uploadImage(
                $request->file('site_logo'),
                'logo'
            );
        }

        if ($request->hasFile('site_favicon')) {
            $this->deleteOldFile('site_favicon', 'assets/images/favicon');

            $data['site_favicon'] = $this->fileManager->uploadImage(
                $request->file('site_favicon'),
                'favicon'
            );
        }


        $this->updateSettingsInDatabase($data);
        Cache::forget('settings');

        return redirect()->back()->with('success', 'Settings updated successfully.');
    }

    private function validateFraud(Request $request)
    {
        $request->validate([
            'email_verification_enabled' => 'required|boolean',
            'registration_enabled' => 'required|boolean',
            'vpn_ban_enabled' => 'required|boolean',
            'country_ban_enabled' => 'required|boolean',
            'one_device_registration_only' => 'required|boolean',
            'vpn_detection_enabled' => 'required|boolean',
            'temporary_email_ban_enabled' => 'required|boolean',
            'captcha_enabled' => 'required|boolean',
        ]);
    }

    private function validateSettings(Request $request)
    {
        $request->validate([
            'site_currency_symbol' => 'required|string',
            'active_template' => 'required|string',
            'referral_percentage' => 'required|numeric',
            'site_logo' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:500',
            'site_favicon' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:500',
            'seo_description' => 'required|string',
            'seo_keywords' => 'required|string',
            'google_analytics_key' => 'nullable|string',
            'proxycheck_api_key' => 'nullable|string',
        ]);
    }

    private function getSettingsFromRequest(Request $request, array $keys)
    {
        return collect($request->only($keys));
    }

    private function updateSettingsInDatabase(array $data)
    {
        foreach ($this->generalSettings as $key) {
            if (array_key_exists($key, $data)) {
                Setting::updateOrCreate(['name' => $key], ['value' => $data[$key]]);
            }
        }
    }

    private function deleteOldFile(string $key, string $directory)
    {
        $oldFilePath = Setting::where('name', $key)->value('value');

        if ($oldFilePath) {
            $oldFilePath = 'public/' . $directory . '/' . basename($oldFilePath);

            if (Storage::exists($oldFilePath)) {
                Storage::delete($oldFilePath);
            }
        }
    }
}
