<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\LeaderboardSettings;
use App\Services\LeaderboardService;
use App\Models\Track;
use App\Models\User;
use App\Models\WithdrawalHistory;
use Carbon\Carbon;
use App\Services\HomeService;


class DashboardController extends Controller
{
    protected $homeService;
    protected $leaderboardService;

    public function __construct(HomeService $homeService, LeaderboardService $leaderboardService)
    {
        $this->homeService = $homeService;
        $this->leaderboardService = $leaderboardService;
    }

    //Home Controller
    public function index(Request $request)
    {
        $data = $this->homeService->getHomeData($request);

        if(auth()->check() === false){

            // totalUsers
            $totalUsers = User::count();
            // total Offers
            $totalOffers = Track::count();
            //total Paid
            $totalPaid = WithdrawalHistory::sum('amount');

            return view($data['activeTemplate'] . '.welcome', [
                'totalUsers' => $totalUsers,
                'totalOffers' => $totalOffers,
                'totalPaid' => $totalPaid
            ]);
        }

        return view($data['activeTemplate'] . '.home', [
            'isVPNDetected' => $data['isVPNDetected'],
            'offer_networks' => $data['offer_networks'],
            'survey_networks' => $data['survey_networks'],
            'device' => $data['device'],
            'adgateOffers' => $data['adgateOffers'],
            'ogadsOffers' => $data['ogadsOffers'],
            'notikOffers' => $data['notikOffers'],
        ]);
    }


    public function checkBalanceIncrease(Request $request)
    {
        $user = auth()->user();

        // Get the last known balance from session or cache
        $lastBalance = session('last_balance', $user->balance);

        // Check if the balance has increased
        if ($user->balance > $lastBalance) {
            session(['last_balance' => $user->balance]); // Update session balance
            return response()->json(['increased' => true]);
        }

        session(['last_balance' => $user->balance]); // Update session balance
        return response()->json(['increased' => false]);
    }


    //Leaderboard Controller
    public function leaderboard()
    {
        $activeTemplate = getActiveTemplate();

        $settings = LeaderboardSettings::first();

        if (!$settings || !$settings->status) {
            abort(403, 'Leaderboard is currently disabled.');
        }

        $topPrizes = json_decode($settings->top_ranks_prizes, true);
        $totalTopPrizes = array_sum($topPrizes);

        $totalPrizePool = $settings->total_prize;

        // Define the period based on the duration set in the leaderboard settings
        $period = $this->leaderboardService->getPeriod($settings->duration);
        $periodEnd = $this->leaderboardService->getPeriod($settings->duration)[1];
        $timeLeft = sprintf("%d:%s", Carbon::now()->diffInDays($periodEnd), Carbon::now()->diff($periodEnd)->format('%H:%I:%S'));

        // Calculate today's earnings
        $earnedToday = Track::where('uid', Auth::user()->uid)
            ->whereDate('created_at', Carbon::today())
            ->where('status', 1)
            ->sum('reward');


        $users = $this->leaderboardService->getActiveUsersWithRewards($period, $settings->number_of_users);

        $distributionMethod = $settings->distribution_method;

        // Calculate potential prizes for each user
        $potentialPrizes = $this->leaderboardService->calculatePrizes($users, $topPrizes, $totalPrizePool, $distributionMethod);

        return view($activeTemplate . '.leaderboard', compact('users', 'potentialPrizes', 'earnedToday', 'totalPrizePool', 'timeLeft'));
    }



}
