<?php

use App\Models\Setting;
use GeoIp2\Database\Reader;
use Anhskohbo\NoCaptcha\Facades\NoCaptcha;
use Jenssegers\Agent\Agent;


function loadSettings()
{
    static $settingsCache = null;

    if ($settingsCache === null) {
        $settingsCache = Setting::whereIn('name', [
            'active_template',
            'site_name',
            'site_logo',
            'site_favicon',
            'site_currency_symbol',
            'seo_description',
            'seo_keywords',
            'google_analytics_key',
            'registration_enabled',
            'captcha_enabled',
            'referral_percentage',
            'adgate_status_enabled',
            'ogads_status_enabled',
            'country_ban_enabled',
            'vpn_ban_enabled',
            'vpn_detection_enabled',
            'notik_app_id',
            'notik_pub_id',
            'notik_api_key',
            'notik_api_key_secret',
            'notik_rate',
            'notik_status_enabled',
        ])->pluck('value', 'name')->toArray();
    }

    return $settingsCache;
}

function getSetting($key, $default = null)
{
    $settings = loadSettings();
    return $settings[$key] ?? $default;
}

// Consolidated settings accessors
function get_current_template()
{
    return getSetting('active_template', 'garnet');
}

function siteName()
{
    return getSetting('site_name');
}

function siteLogo()
{
    return getSetting('site_logo');
}

function siteFav()
{
    return getSetting('site_favicon');
}

function siteSymbol()
{
    return getSetting('site_currency_symbol');
}

function seoDescription()
{
    return getSetting('seo_description', '');
}

function seoKeywords()
{
    return getSetting('seo_keywords', '');
}

function googleAnalyticsKey()
{
    return getSetting('google_analytics_key');
}

function isRegistrationEnabled()
{
    return getSetting('registration_enabled') === '1';
}

function isCaptchaEnabled()
{
    return getSetting('captcha_enabled') === '1';
}

function ReferralCommission()
{
    return getSetting('referral_percentage', 0);
}

// Template utilities
function getActiveTemplate()
{
    $activeTemplate = view()->shared('activeTemplate');
    return str_replace('/', '.', $activeTemplate);
}

function get_available_templates()
{
    $templatePath = resource_path('views/templates');
    return array_map(function ($dir) use ($templatePath) {
        return basename($dir);
    }, array_filter(glob($templatePath . '/*'), 'is_dir'));
}

// GeoIP utility
function getCountryCode($ip)
{
    try {
        $reader = new Reader(storage_path('app/geoip/GeoLite2-Country.mmdb'));
        return $reader->country($ip)->country->isoCode ?? null;
    } catch (\Exception $e) {
        return null;
    }
}

function getFullCountryName($iso)
{
    static $countries = null;

    if ($countries === null) {
        $countries = json_decode(file_get_contents(storage_path('app/geoip/countries.json')), true);
        // Convert to associative array for faster lookups
        $countries = array_column($countries, 'name', 'code');
    }

    return $countries[strtoupper($iso)] ?? 'Unknown Country';
}



// CAPTCHA rendering
function renderCaptcha()
{
    if (isCaptchaEnabled()) {
        return '<div class="d-flex justify-content-center mb-2">' .
            '<div>' . NoCaptcha::display() . '</div>' .
            '</div>';
    }
    return '';
}

// Device detection
function detectDevicePlatform()
{
    $agent = new Agent();
    return match (true) {
        $agent->isAndroidOS() => 'android',
        $agent->is('iPhone') => 'ios',
        $agent->isDesktop() => 'desktop',
        $agent->is('Macintosh') => 'mac os',
        default => 'unknown',
    };
}

// OGADS and ADGATE API status
function isNotikApiEnabled()
{
    return getSetting('notik_status_enabled') === '1';
}

function isOgadsApiEnabled()
{
    return getSetting('ogads_status_enabled') === '1';
}

function isAdgateApiEnabled()
{
    return getSetting('adgate_status_enabled') === '1';
}

function isCountryBanEnabled()
{
    return (int) getSetting('country_ban_enabled') === 1;
}
function isVpnBanEnabled()
{
    return (int) getSetting('vpn_ban_enabled') === 1;
}

function isVpnDetectionEnabled()
{
    return (int) getSetting('vpn_detection_enabled') === 0;
}
