<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Str;

class User extends Authenticatable implements MustVerifyEmail
{
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'uid',
        'email',
        'password',
        'balance',
        'ip',
        'last_login_ip',
        'user_agent',
        'country_code',
        'gender',
        'date_of_birth',
        'avatar',
        'referral_code',
        'email_verified_at',
        'invited_by',
        'status',
    ];
    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    protected static function booted()
    {
        static::creating(function ($user) {
            if (empty($user->uid)) {
                $user->uid = self::generateUniqueUid();
            }

            if (empty($user->referral_code)) {
                $user->referral_code = self::generateUniqueReferralCode();
            }
        });
    }

    private static function generateUniqueUid($length = 12)
    {
        $firstChar = strtoupper(mb_substr(SiteName(), 0, 1));
        // Adjust the length for the numeric part
        $numericLength = $length - 1; // 1 for the first character

        do {
            $numericString = $firstChar . self::generateRandomDigits($numericLength);
        } while (self::where('uid', $numericString)->exists());

        return $numericString;
    }

    private static function generateRandomDigits($length)
    {
        // Generate a numeric string of the specified length
        return implode('', array_map(function () {
            return mt_rand(0, 9);
        }, range(1, $length)));
    }

    private static function generateUniqueReferralCode($length = 5)
    {
        do {
            $referralCode = Str::random($length);
        } while (self::where('referral_code', $referralCode)->exists());

        return $referralCode;
    }



    public function invitedBy()
    {
        return $this->belongsTo(User::class, 'invited_by', 'id');
    }
}
