<?php

namespace App\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Network;
use App\Services\OfferService;
use App\Services\VPNDetectionService;

class HomeService
{
    protected $offerService;
    protected $vpnService;

    // Type-hinting both services for better clarity
    public function __construct(OfferService $offerService, VPNDetectionService $vpnService)
    {
        $this->offerService = $offerService;
        $this->vpnService = $vpnService;
    }


    public function getHomeData(Request $request): array
    {
        $activeTemplate = getActiveTemplate();
        $user = Auth::user();
        $userUid = $user ? $user->uid : null;
        $device = detectDevicePlatform();
        $isVPNDetected = $this->vpnService->isVPN($request->ip());

        $data = [
            'activeTemplate' => $activeTemplate,
            'isVPNDetected' => $isVPNDetected,
            'device' => $device,
            'offer_networks' => [],
            'survey_networks' => [],
            'adgateOffers' => [],
            'ogadsOffers' => [],
            'notikOffers' => [],
        ];

        // Only proceed to fetch offers if a VPN is not detected
        if (!$isVPNDetected) {
            $userCountry = getCountryCode($request->ip()) ?? 'US';

            $data['notikOffers'] = $this->offerService->getOffers('notik', $userCountry, 20);
            $data['adgateOffers'] = $this->offerService->getOffers('adgatemedia', $userCountry, 20);
            $data['ogadsOffers'] = $this->offerService->fetchOgadsOffers($request);

            // Get offer and survey networks
            $data['offer_networks'] = $this->getNetworksByType('offer', $userUid);
            $data['survey_networks'] = $this->getNetworksByType('survey', $userUid);
        }

        return $data;
    }



    // Method to get networks by type, use type hinting
    private function getNetworksByType(string $type, ?string $userUid): \Illuminate\Support\Collection
    {
        return Network::select('network_name', 'network_image', 'network_description', 'network_rating', 'offerwall_url')
            ->where('network_status', 1)
            ->where('network_type', $type)
            ->get()
            ->map(function ($network) use ($userUid) {
                // Only replace UID if it's not null
                if ($userUid) {
                    $network->offerwall_url = $this->replaceUidInUrl($network->offerwall_url, $userUid);
                }
                return $network;
            });
    }

    // Replaces UID in URL, type-hinting return value
    private function replaceUidInUrl(string $url, ?string $userUid): string
    {
        return $userUid ? str_replace('[uid]', $userUid, $url) : $url;
    }
}
